<?php
/**
 * Enqueue script and styles for child theme
 */
function woodmart_child_enqueue_styles() {
	wp_enqueue_style( 'child-style', get_stylesheet_directory_uri() . '/style.css', array( 'woodmart-style' ), woodmart_get_theme_info( 'Version' ) );
}
add_action( 'wp_enqueue_scripts', 'woodmart_child_enqueue_styles', 10010 );


function set_default_shipping_method( $package ) {
    $free_shipping_available = false;

    // Check if free shipping is available
    foreach ( WC()->shipping()->get_shipping_methods() as $shipping_method ) {
        if ( 'free_shipping' === $shipping_method->id ) {
            $free_shipping_available = true;
            break;
        }
    }

    // If free shipping is available, set it as the default
    if ( $free_shipping_available ) {
        WC()->session->set( 'chosen_shipping_methods', array( 'free_shipping' ) );
    }
}

// Hook into WooCommerce before calculate shipping action
add_action( 'woocommerce_before_calculate_totals', 'set_default_shipping_method' );


function apply_category_quantity_discount() {
    $category_slug = 'charms'; // Replace with the actual slug of the category
    $discount_rate = .5; // Discount per item, change as needed

    // Check if the cart is not empty
    if ( ! WC()->cart->is_empty() ) {
        $total_quantity = 0;

        // Loop through cart items
        foreach ( WC()->cart->get_cart() as $cart_item ) {
            $product_id = $cart_item['product_id'];
            $product_categories = get_the_terms( $product_id, 'product_cat' );

            // Check if the product belongs to the specified category
            if ( $product_categories ) {
                foreach ( $product_categories as $category ) {
                    if ( $category->slug === $category_slug ) {
                        $total_quantity += $cart_item['quantity'];
                        break;
                    }
                }
            }
        }

        // Apply discount based on the total quantity
        if ( $total_quantity >= 20 && $total_quantity <= 24  ) {
            $discount_amount = 10; // Discount for 10 or more items
        } elseif ( $total_quantity >= 15 && $total_quantity <= 19  ) {
            $discount_amount = 7.5; // Discount for 10 or more items
        } elseif ( $total_quantity >= 10 && $total_quantity <= 14  ) {
            $discount_amount = 5; // Discount for 10 or more items
        } elseif ( $total_quantity >= 5 && $total_quantity <= 9  ) {
            $discount_amount = 2.5; // Discount for 5 or more items
        }

        // Apply the discount to the cart
        if ( isset( $discount_amount ) ) {
            $cart_discount = $discount_amount ;
            WC()->cart->add_fee( 'Charms Discount', -$cart_discount );
        }
    }
}

// Hook into WooCommerce cart calculations
add_action( 'woocommerce_before_calculate_totals', 'apply_category_quantity_discount' );

// -----------------------------------------------------------------------------------

// Add custom fields to registration form
add_action('woocommerce_register_form_start', 'add_custom_registration_fields');

function add_custom_registration_fields() {
    ?>
    <p class="form-row form-row-first">
        <label for="reg_billing_first_name"><?php _e('First Name', 'woocommerce'); ?><span class="required">*</span></label>
        <input type="text" class="input-text" name="billing_first_name" id="reg_billing_first_name" value="<?php if (!empty($_POST['billing_first_name'])) echo esc_attr($_POST['billing_first_name']); ?>" required />
    </p>
    <p class="form-row form-row-last">
        <label for="reg_billing_last_name"><?php _e('Last Name', 'woocommerce'); ?><span class="required">*</span></label>
        <input type="text" class="input-text" name="billing_last_name" id="reg_billing_last_name" value="<?php if (!empty($_POST['billing_last_name'])) echo esc_attr($_POST['billing_last_name']); ?>" required />
    </p>
    <p class="form-row form-row-wide">
        <label for="reg_billing_phone"><?php _e('Phone', 'woocommerce'); ?><span class="required">*</span></label>
        <input type="tel" class="input-text" name="billing_phone" id="reg_billing_phone" value="<?php if (!empty($_POST['billing_phone'])) echo esc_attr($_POST['billing_phone']); ?>" required />
    </p>
    <p class="form-row form-row-wide">
        <label for="reg_birthdate"><?php _e('Date of Birth', 'woocommerce'); ?><span class="required">*</span></label>
        <input type="date" class="input-text" name="birthdate" id="reg_birthdate" value="<?php if (!empty($_POST['birthdate'])) echo esc_attr($_POST['birthdate']); ?>" required />
    </p>


    <?php
}

// Validate and save custom fields on registration
add_action('woocommerce_register_post', 'validate_and_save_custom_registration_fields', 10, 3);

function validate_and_save_custom_registration_fields($username, $email, $validation_errors) {
     if (isset($_POST['billing_first_name']) && empty($_POST['billing_first_name'])) {
        $validation_errors->add('billing_first_name_error', __('Please enter your first name.', 'woocommerce'));
    }

    if (isset($_POST['billing_last_name']) && empty($_POST['billing_last_name'])) {
        $validation_errors->add('billing_last_name_error', __('Please enter your last name.', 'woocommerce'));
    }

    if (isset($_POST['billing_phone']) && empty($_POST['billing_phone'])) {
        $validation_errors->add('billing_phone_error', __('Please enter your billing phone number.', 'woocommerce'));
    }
	
	if (isset($_POST['birthdate']) && empty($_POST['birthdate'])) {
        $validation_errors->add('birthdate_error', __('Please enter your date of birth.', 'woocommerce'));
    }
    return $validation_errors;
}

// Save custom fields to user meta
add_action('woocommerce_created_customer', 'save_custom_registration_fields');

function save_custom_registration_fields($customer_id) {
	if (isset($_POST['billing_first_name'])) {
        update_user_meta($customer_id, 'first_name', sanitize_text_field($_POST['billing_first_name']));
    }

    if (isset($_POST['billing_last_name'])) {
        update_user_meta($customer_id, 'last_name', sanitize_text_field($_POST['billing_last_name']));
    }

    if (isset($_POST['billing_phone'])) {
        update_user_meta($customer_id, 'billing_phone', sanitize_text_field($_POST['billing_phone']));
    }
    if (isset($_POST['birthdate'])) {
        update_user_meta($customer_id, 'birthdate', sanitize_text_field($_POST['birthdate']));
    }
}

// Display custom fields on the user profile page
add_action('show_user_profile', 'display_custom_profile_fields');
add_action('edit_user_profile', 'display_custom_profile_fields');

function display_custom_profile_fields($user) {
    ?>
    <h3><?php _e('Additional Information', 'woocommerce'); ?></h3>

    <table class="form-table">
        <tr>
            <th><label for="birthdate"><?php _e('Date of Birth', 'woocommerce'); ?></label></th>
            <td>
                <input type="text" name="birthdate" id="birthdate" value="<?php echo esc_attr(get_user_meta($user->ID, 'birthdate', true)); ?>" class="regular-text" />
            </td>
        </tr>

    </table>
    <?php
}

// Save First Name, Last Name, and Billing Phone on user profile update
add_action('personal_options_update', 'save_custom_fields_on_profile_update');
add_action('edit_user_profile_update', 'save_custom_fields_on_profile_update');

function save_custom_fields_on_profile_update($user_id) {
    if (current_user_can('edit_user', $user_id)) {
        update_user_meta($user_id, 'first_name', sanitize_text_field($_POST['billing_first_name']));
        update_user_meta($user_id, 'last_name', sanitize_text_field($_POST['billing_last_name']));
        update_user_meta($user_id, 'billing_phone', sanitize_text_field($_POST['billing_phone']));
    }
}


// Add custom columns to the Users admin page
add_filter('manage_users_columns', 'custom_user_table_columns');

function custom_user_table_columns($columns) {
    $columns['billing_phone'] = __('Phone', 'woocommerce');
    $columns['birthdate'] = __('Date of Birth', 'woocommerce');
    return $columns;
}

// Display data in the custom columns
add_action('manage_users_custom_column', 'custom_user_table_column_content', 10, 3);

function custom_user_table_column_content($value, $column_name, $user_id) {
    switch ($column_name) {
        case 'billing_phone':
            return get_user_meta($user_id, 'billing_phone', true);
        case 'birthdate':
            return get_user_meta($user_id, 'birthdate', true);
        default:
            return $value;
    }
}

// Make custom columns sortable
add_filter('manage_users_sortable_columns', 'custom_user_table_sortable_columns');

function custom_user_table_sortable_columns($sortable_columns) {
    $sortable_columns['billing_phone'] = 'billing_phone';
    $sortable_columns['birthdate'] = 'birthdate';
    return $sortable_columns;
}


// Add Date of Birth field to the My Account page
add_action('woocommerce_edit_account_form', 'add_dob_to_edit_account_form');

function add_dob_to_edit_account_form() {
    $user_id = get_current_user_id();
    $birthdate = get_user_meta($user_id, 'birthdate', true);
    ?>
    <p class="woocommerce-form-row woocommerce-form-row--wide form-row form-row-wide">
        <label for="dob"><?php _e('Date of Birth', 'woocommerce'); ?></label>
        <input type="date" class="input-text" name="dob" id="dob" value="<?php echo esc_attr($birthdate); ?>" />
    </p>
    <?php
}

// Save Date of Birth when the account form is submitted
add_action('woocommerce_save_account_details', 'save_dob_from_account_form');

function save_dob_from_account_form($user_id) {
    if (isset($_POST['dob'])) {
        update_user_meta($user_id, 'birthdate', sanitize_text_field($_POST['dob']));
    }
}



